using System;
using System.Drawing;
using System.Collections;
using System.Windows.Forms;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.BOL;
using gov.va.med.vbecs.GUI;
using gov.va.med.vbecs.GUI.controls;

using BrRules = gov.va.med.vbecs.Common.VbecsBrokenRules.VbecsUser;

namespace gov.va.med.vbecs.admin.GUI
{
	/// <summary>
	/// User detail control. 
	/// </summary>
	public class EditVbecsUserParameters : BaseAdminFormControl
	{		
		#region VbecsUserListViewItem implementation

		private class VbecsUserListViewItem : BaseListViewItem
		{
			private enum UserSubItemsIndex : int
			{
				UserNtLoginId = 1,
				NtUserName,
				Duz,
				VistaUserName,
				Initials,
				UserEmail				
			};

			private static readonly int _numberOfSubItems;
			private readonly VbecsUser _vbecsUser;
			private static readonly Hashtable _allItems;

			static VbecsUserListViewItem()
			{
				_numberOfSubItems = Enum.GetValues( typeof(UserSubItemsIndex) ).Length;
				_allItems = new Hashtable();
			}

			public VbecsUserListViewItem( VbecsUser vbecsUser ) 
				: base( _numberOfSubItems )
			{
				if( vbecsUser == null )
					throw( new ArgumentNullException( "vbecsUser" ) );

				_vbecsUser = vbecsUser;

				SyncWithDataObject();

				lock( typeof( VbecsUserListViewItem ) )
					if( _allItems.ContainsKey( vbecsUser ) )
						_allItems[ vbecsUser ] = this;
					else
						_allItems.Add( vbecsUser, this );
			}

			public override void SyncWithDataObject()
			{
				if( this.Checked != _vbecsUser.IsActive ) // to prevent self-calling
					this.Checked = _vbecsUser.IsActive;

				this.SubItems[ (int)UserSubItemsIndex.Duz ].Text = _vbecsUser.UserDuz;
				this.SubItems[ (int)UserSubItemsIndex.VistaUserName ].Text = _vbecsUser.VistAUserName; 
				this.SubItems[ (int)UserSubItemsIndex.UserNtLoginId ].Text = _vbecsUser.UserNTLoginID;
				this.SubItems[ (int)UserSubItemsIndex.NtUserName ].Text = _vbecsUser.UserName;
				this.SubItems[ (int)UserSubItemsIndex.UserEmail ].Text = _vbecsUser.UserEmail;		
				this.SubItems[ (int)UserSubItemsIndex.Initials ].Text = _vbecsUser.UserInitials;

				SetStatusForeColor( _vbecsUser.IsActive );
			}

			public VbecsUser VbecsUser
			{
				get
				{
					return _vbecsUser;
				}
			}

			public static VbecsUserListViewItem GetListViewItemByVbecsUser( VbecsUser user )
			{
				if( user == null )
					throw( new ArgumentNullException( "user" ) );

				lock( typeof( VbecsUserListViewItem ) )
					return (VbecsUserListViewItem)_allItems[ user ];
			}
		}

		#endregion

		#region 'No Data Found' ListViewItems implementations
		
		/// <summary>
		/// Class NoUsersDefinedListViewItem
		/// </summary>
		protected class NoUsersDefinedListViewItem : NoDataFoundListViewItem
		{
			/// <summary>
			/// Constructor
			/// </summary>
			public NoUsersDefinedListViewItem() 
				: base( StrRes.OtherMsg.MUC03.NoUsersDefinedInVbecs().ResString ) {}
		}

		/// <summary>
		/// Class ThereAreNoActiveUsersListViewItem
		/// </summary>
		protected class ThereAreNoActiveUsersListViewItem : NoDataFoundListViewItem
		{
			/// <summary>
			/// Constructor
			/// </summary>
			public ThereAreNoActiveUsersListViewItem() 
				: base( StrRes.OtherMsg.MUC03.NoActiveUsersToDisplay().ResString ) {}
		}

		#endregion

		private Hashtable _vistAUsers;
		private BusinessObjectSortedHashList _vbecsUsers;
		private VbecsUser _selectedUser;
		private BusinessObjectGuiValidator _vbecsUserValidator;

		private System.Windows.Forms.GroupBox grpUserIdentification;
		private VbecsButton btnUserSave;
		private System.Windows.Forms.CheckBox cbActiveVbecsUser;
		private System.Windows.Forms.GroupBox grpAdditionalInfo;
		private System.Windows.Forms.TextBox txtUserEmailAddress;
		private System.Windows.Forms.Label lblUserEmailAddress;
		private System.Windows.Forms.GroupBox grpNTUser;
		private System.Windows.Forms.ComboBox cmbNTUserID;
		private System.Windows.Forms.Label lblNTUeserID;
		private System.Windows.Forms.Label lblNTUserName;
		private System.Windows.Forms.GroupBox grpVistAUser;
		private System.Windows.Forms.TextBox txtVistaDuz;
		private System.Windows.Forms.Label lblVistADUZ;
		private System.Windows.Forms.Label lblVistAUserName;
		private VbecsButton btnVistAUsers;
		private VbecsButton btnUserClear;
		private System.Windows.Forms.GroupBox grpUserConfiguration;
		private System.Windows.Forms.ListView lvUsers;
		private System.Windows.Forms.CheckBox cbShowInactiveUsers;
		private System.Windows.Forms.ColumnHeader colDUZ;
		private System.Windows.Forms.ColumnHeader colVistAName;
		private System.Windows.Forms.ColumnHeader colUserID;
		private System.Windows.Forms.ColumnHeader colUserName;
		private System.Windows.Forms.ColumnHeader colEmail;
		private System.Windows.Forms.ColumnHeader colInitials;
		private System.Windows.Forms.TextBox txtVistaUserName;
		private System.Windows.Forms.TextBox txtUserNtName;
		private System.Windows.Forms.TextBox txtUserInitials;
		private gov.va.med.vbecs.admin.GUI.EditVbecsUserDivisionalAccess editVbecsUserDivisionalAccess;
		private System.Windows.Forms.Label lblUserInitials;
		private System.Windows.Forms.ColumnHeader colActive;
		private VbecsButton btnWindowsUsers;
		private bool _formInitialized;

        /// <summary>
        /// Defect 251166 - need a seperate errProvider since UserName is non-editable, this means
        ///  our normal brokenrules routines won't set it properly or clear it
        /// </summary>
        private gov.va.med.vbecs.GUI.controls.VbecsErrorProvider erpUserNameProvider;

		/// <summary> 
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;

		/// <summary>
		/// Constructor
		/// </summary>
		public EditVbecsUserParameters() : base()
		{
			// This call is required by the Windows.Forms Form Designer.
			InitializeComponent();			
		
			_selectedUser = null;	
			SetupVbecsUserValidator();

			SetButtonsCaptions();
			LogonUser.NotDefined = true;

			_formInitialized = false;
		}

		private void SetButtonsCaptions()
		{
			btnUserClear.Text = "&Clear";
			btnUserSave.Text = "&Save";
			btnVistAUsers.Text = btnWindowsUsers.Text = "...";
		}

		private void SetupVbecsUserValidator()
		{
			_vbecsUserValidator = new BusinessObjectGuiValidator( this.erpMainProvider, BrRules.FormClassName );
				
			// CR1902 Added "BrRules.DivisionDuzMismatch" for new business rule
			_vbecsUserValidator.SetControlBindings( txtVistaDuz, BrRules.UserDuzNotSet, BrRules.UserDuzInvalid, BrRules.UserDuzIsNotUnique, BrRules.DivisionDuzMismatch );
			_vbecsUserValidator.SetControlBindings( cmbNTUserID, BrRules.UserNtLoginIdNotSet, BrRules.UserNtLoginIdInvalid, BrRules.UserNtLoginIsNotUnique );
			_vbecsUserValidator.SetControlBindings( txtVistaUserName, BrRules.VistaUserNameNotSet, BrRules.VistaUserNameTooLong, BrRules.VistaUserNameIsNotUnique );
			_vbecsUserValidator.SetControlBindings( txtUserEmailAddress, BrRules.UserEmailInvalid );
			_vbecsUserValidator.SetControlBindings( txtUserInitials, BrRules.UserInitialsNotSet, BrRules.UserInitialsAreTooLong, BrRules.UserInitialsAreNotUnique );
			_vbecsUserValidator.SetControlBindings( txtUserNtName, BrRules.UserNtNameNotSet, BrRules.UserNameTooLong );
		}

		/// <summary> 
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#region Component Designer generated code
		/// <summary> 
		/// Required method for Designer support - do not modify 
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
            this.erpUserNameProvider = new gov.va.med.vbecs.GUI.controls.VbecsErrorProvider();
            this.txtUserNtName = new System.Windows.Forms.TextBox();
            this.grpUserIdentification = new System.Windows.Forms.GroupBox();
            this.grpVistAUser = new System.Windows.Forms.GroupBox();
            this.txtVistaDuz = new System.Windows.Forms.TextBox();
            this.lblVistADUZ = new System.Windows.Forms.Label();
            this.txtVistaUserName = new System.Windows.Forms.TextBox();
            this.lblVistAUserName = new System.Windows.Forms.Label();
            this.btnVistAUsers = new gov.va.med.vbecs.GUI.controls.VbecsButton();
            this.grpNTUser = new System.Windows.Forms.GroupBox();
            this.btnWindowsUsers = new gov.va.med.vbecs.GUI.controls.VbecsButton();
            this.cmbNTUserID = new System.Windows.Forms.ComboBox();
            this.lblNTUeserID = new System.Windows.Forms.Label();
            this.lblNTUserName = new System.Windows.Forms.Label();
            this.grpAdditionalInfo = new System.Windows.Forms.GroupBox();
            this.txtUserInitials = new System.Windows.Forms.TextBox();
            this.lblUserInitials = new System.Windows.Forms.Label();
            this.txtUserEmailAddress = new System.Windows.Forms.TextBox();
            this.lblUserEmailAddress = new System.Windows.Forms.Label();
            this.cbActiveVbecsUser = new System.Windows.Forms.CheckBox();
            this.btnUserSave = new gov.va.med.vbecs.GUI.controls.VbecsButton();
            this.btnUserClear = new gov.va.med.vbecs.GUI.controls.VbecsButton();
            this.grpUserConfiguration = new System.Windows.Forms.GroupBox();
            this.lvUsers = new System.Windows.Forms.ListView();
            this.colActive = ((System.Windows.Forms.ColumnHeader)(new System.Windows.Forms.ColumnHeader()));
            this.colUserID = ((System.Windows.Forms.ColumnHeader)(new System.Windows.Forms.ColumnHeader()));
            this.colUserName = ((System.Windows.Forms.ColumnHeader)(new System.Windows.Forms.ColumnHeader()));
            this.colDUZ = ((System.Windows.Forms.ColumnHeader)(new System.Windows.Forms.ColumnHeader()));
            this.colVistAName = ((System.Windows.Forms.ColumnHeader)(new System.Windows.Forms.ColumnHeader()));
            this.colInitials = ((System.Windows.Forms.ColumnHeader)(new System.Windows.Forms.ColumnHeader()));
            this.colEmail = ((System.Windows.Forms.ColumnHeader)(new System.Windows.Forms.ColumnHeader()));
            this.cbShowInactiveUsers = new System.Windows.Forms.CheckBox();
            this.editVbecsUserDivisionalAccess = new gov.va.med.vbecs.admin.GUI.EditVbecsUserDivisionalAccess();
            ((System.ComponentModel.ISupportInitialize)(this.erpMainProvider)).BeginInit();
            ((System.ComponentModel.ISupportInitialize)(this.erpUserNameProvider)).BeginInit();
            this.grpUserIdentification.SuspendLayout();
            this.grpVistAUser.SuspendLayout();
            this.grpNTUser.SuspendLayout();
            this.grpAdditionalInfo.SuspendLayout();
            this.grpUserConfiguration.SuspendLayout();
            this.SuspendLayout();
            // 
            // erpUserNameProvider
            // 
            this.erpUserNameProvider.ContainerControl = this;
            this.erpUserNameProvider.Icon = gov.va.med.vbecs.GUI.controls.VbecsIcon.Informational.Icon;
            // 
            // txtUserNtName
            // 
            this.txtUserNtName.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.erpUserNameProvider.SetIconAlignment(this.txtUserNtName, System.Windows.Forms.ErrorIconAlignment.MiddleLeft);
            this.erpMainProvider.SetIconAlignment(this.txtUserNtName, System.Windows.Forms.ErrorIconAlignment.MiddleLeft);
            this.txtUserNtName.Location = new System.Drawing.Point(82, 40);
            this.txtUserNtName.Name = "txtUserNtName";
            this.txtUserNtName.ReadOnly = true;
            this.txtUserNtName.Size = new System.Drawing.Size(168, 20);
            this.txtUserNtName.TabIndex = 30;
            // 
            // grpUserIdentification
            // 
            this.grpUserIdentification.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.grpUserIdentification.Controls.Add(this.grpVistAUser);
            this.grpUserIdentification.Controls.Add(this.grpNTUser);
            this.grpUserIdentification.Controls.Add(this.grpAdditionalInfo);
            this.grpUserIdentification.Location = new System.Drawing.Point(0, 0);
            this.grpUserIdentification.Name = "grpUserIdentification";
            this.grpUserIdentification.Size = new System.Drawing.Size(568, 136);
            this.grpUserIdentification.TabIndex = 12;
            this.grpUserIdentification.TabStop = false;
            this.grpUserIdentification.Text = "User Identification";
            // 
            // grpVistAUser
            // 
            this.grpVistAUser.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.grpVistAUser.Controls.Add(this.txtVistaDuz);
            this.grpVistAUser.Controls.Add(this.lblVistADUZ);
            this.grpVistAUser.Controls.Add(this.txtVistaUserName);
            this.grpVistAUser.Controls.Add(this.lblVistAUserName);
            this.grpVistAUser.Controls.Add(this.btnVistAUsers);
            this.grpVistAUser.Location = new System.Drawing.Point(270, 16);
            this.grpVistAUser.Name = "grpVistAUser";
            this.grpVistAUser.Size = new System.Drawing.Size(292, 68);
            this.grpVistAUser.TabIndex = 0;
            this.grpVistAUser.TabStop = false;
            this.grpVistAUser.Text = "VistA User";
            // 
            // txtVistaDuz
            // 
            this.txtVistaDuz.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.erpMainProvider.SetIconAlignment(this.txtVistaDuz, System.Windows.Forms.ErrorIconAlignment.MiddleLeft);
            this.txtVistaDuz.Location = new System.Drawing.Point(82, 16);
            this.txtVistaDuz.Name = "txtVistaDuz";
            this.txtVistaDuz.ReadOnly = true;
            this.txtVistaDuz.Size = new System.Drawing.Size(178, 20);
            this.txtVistaDuz.TabIndex = 0;
            // 
            // lblVistADUZ
            // 
            this.lblVistADUZ.Location = new System.Drawing.Point(8, 16);
            this.lblVistADUZ.Name = "lblVistADUZ";
            this.lblVistADUZ.Size = new System.Drawing.Size(64, 20);
            this.lblVistADUZ.TabIndex = 29;
            this.lblVistADUZ.Text = "VistA DUZ*";
            this.lblVistADUZ.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // txtVistaUserName
            // 
            this.txtVistaUserName.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.erpMainProvider.SetIconAlignment(this.txtVistaUserName, System.Windows.Forms.ErrorIconAlignment.MiddleLeft);
            this.txtVistaUserName.Location = new System.Drawing.Point(82, 40);
            this.txtVistaUserName.Name = "txtVistaUserName";
            this.txtVistaUserName.ReadOnly = true;
            this.txtVistaUserName.Size = new System.Drawing.Size(206, 20);
            this.txtVistaUserName.TabIndex = 2;
            // 
            // lblVistAUserName
            // 
            this.lblVistAUserName.Location = new System.Drawing.Point(8, 40);
            this.lblVistAUserName.Name = "lblVistAUserName";
            this.lblVistAUserName.Size = new System.Drawing.Size(68, 20);
            this.lblVistAUserName.TabIndex = 3;
            this.lblVistAUserName.Text = "User Name*";
            this.lblVistAUserName.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // btnVistAUsers
            // 
            this.btnVistAUsers.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
            this.btnVistAUsers.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Bold, System.Drawing.GraphicsUnit.Point, ((byte)(0)));
            this.btnVistAUsers.Image = null;
            this.btnVistAUsers.Location = new System.Drawing.Point(264, 16);
            this.btnVistAUsers.Name = "btnVistAUsers";
            this.btnVistAUsers.Size = new System.Drawing.Size(24, 22);
            this.btnVistAUsers.TabIndex = 1;
            this.btnVistAUsers.Click += new System.EventHandler(this.btnVistAUsers_Click);
            // 
            // grpNTUser
            // 
            this.grpNTUser.Controls.Add(this.btnWindowsUsers);
            this.grpNTUser.Controls.Add(this.txtUserNtName);
            this.grpNTUser.Controls.Add(this.cmbNTUserID);
            this.grpNTUser.Controls.Add(this.lblNTUeserID);
            this.grpNTUser.Controls.Add(this.lblNTUserName);
            this.grpNTUser.Location = new System.Drawing.Point(8, 16);
            this.grpNTUser.Name = "grpNTUser";
            this.grpNTUser.Size = new System.Drawing.Size(256, 68);
            this.grpNTUser.TabIndex = 1;
            this.grpNTUser.TabStop = false;
            this.grpNTUser.Text = "NT User";
            // 
            // btnWindowsUsers
            // 
            this.btnWindowsUsers.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
            this.btnWindowsUsers.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Bold, System.Drawing.GraphicsUnit.Point, ((byte)(0)));
            this.btnWindowsUsers.Image = null;
            this.btnWindowsUsers.Location = new System.Drawing.Point(226, 16);
            this.btnWindowsUsers.Name = "btnWindowsUsers";
            this.btnWindowsUsers.Size = new System.Drawing.Size(24, 22);
            this.btnWindowsUsers.TabIndex = 31;
            this.btnWindowsUsers.Click += new System.EventHandler(this.btnWindowsUsers_Click);
            // 
            // cmbNTUserID
            // 
            this.cmbNTUserID.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.cmbNTUserID.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;
            this.erpMainProvider.SetIconAlignment(this.cmbNTUserID, System.Windows.Forms.ErrorIconAlignment.MiddleLeft);
            this.cmbNTUserID.Location = new System.Drawing.Point(82, 16);
            this.cmbNTUserID.Name = "cmbNTUserID";
            this.cmbNTUserID.Size = new System.Drawing.Size(140, 21);
            this.cmbNTUserID.TabIndex = 0;
            this.cmbNTUserID.SelectedIndexChanged += new System.EventHandler(this.cmbNTUserID_SelectedIndexChanged);
            // 
            // lblNTUeserID
            // 
            this.lblNTUeserID.Location = new System.Drawing.Point(8, 16);
            this.lblNTUeserID.Name = "lblNTUeserID";
            this.lblNTUeserID.Size = new System.Drawing.Size(60, 20);
            this.lblNTUeserID.TabIndex = 29;
            this.lblNTUeserID.Text = "User ID*";
            this.lblNTUeserID.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // lblNTUserName
            // 
            this.lblNTUserName.Location = new System.Drawing.Point(8, 40);
            this.lblNTUserName.Name = "lblNTUserName";
            this.lblNTUserName.Size = new System.Drawing.Size(68, 20);
            this.lblNTUserName.TabIndex = 3;
            this.lblNTUserName.Text = "User Name";
            this.lblNTUserName.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // grpAdditionalInfo
            // 
            this.grpAdditionalInfo.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.grpAdditionalInfo.Controls.Add(this.txtUserInitials);
            this.grpAdditionalInfo.Controls.Add(this.lblUserInitials);
            this.grpAdditionalInfo.Controls.Add(this.txtUserEmailAddress);
            this.grpAdditionalInfo.Controls.Add(this.lblUserEmailAddress);
            this.grpAdditionalInfo.Location = new System.Drawing.Point(8, 84);
            this.grpAdditionalInfo.Name = "grpAdditionalInfo";
            this.grpAdditionalInfo.Size = new System.Drawing.Size(554, 44);
            this.grpAdditionalInfo.TabIndex = 2;
            this.grpAdditionalInfo.TabStop = false;
            this.grpAdditionalInfo.Text = "Additional Info";
            // 
            // txtUserInitials
            // 
            this.erpMainProvider.SetIconAlignment(this.txtUserInitials, System.Windows.Forms.ErrorIconAlignment.MiddleLeft);
            this.txtUserInitials.Location = new System.Drawing.Point(344, 16);
            this.txtUserInitials.MaxLength = 3;
            this.txtUserInitials.Name = "txtUserInitials";
            this.txtUserInitials.Size = new System.Drawing.Size(62, 20);
            this.txtUserInitials.TabIndex = 7;
            this.txtUserInitials.TextChanged += new System.EventHandler(this.DataEntryFields_DataChanged);
            // 
            // lblUserInitials
            // 
            this.lblUserInitials.Location = new System.Drawing.Point(270, 16);
            this.lblUserInitials.Name = "lblUserInitials";
            this.lblUserInitials.Size = new System.Drawing.Size(64, 20);
            this.lblUserInitials.TabIndex = 6;
            this.lblUserInitials.Text = "Initials*";
            this.lblUserInitials.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // txtUserEmailAddress
            // 
            this.erpMainProvider.SetIconAlignment(this.txtUserEmailAddress, System.Windows.Forms.ErrorIconAlignment.MiddleLeft);
            this.txtUserEmailAddress.Location = new System.Drawing.Point(82, 16);
            this.txtUserEmailAddress.MaxLength = 50;
            this.txtUserEmailAddress.Name = "txtUserEmailAddress";
            this.txtUserEmailAddress.Size = new System.Drawing.Size(172, 20);
            this.txtUserEmailAddress.TabIndex = 0;
            this.txtUserEmailAddress.TextChanged += new System.EventHandler(this.DataEntryFields_DataChanged);
            // 
            // lblUserEmailAddress
            // 
            this.lblUserEmailAddress.Location = new System.Drawing.Point(8, 16);
            this.lblUserEmailAddress.Name = "lblUserEmailAddress";
            this.lblUserEmailAddress.Size = new System.Drawing.Size(64, 20);
            this.lblUserEmailAddress.TabIndex = 4;
            this.lblUserEmailAddress.Text = "E-mail";
            this.lblUserEmailAddress.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // cbActiveVbecsUser
            // 
            this.cbActiveVbecsUser.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
            this.cbActiveVbecsUser.CheckAlign = System.Drawing.ContentAlignment.MiddleRight;
            this.cbActiveVbecsUser.Location = new System.Drawing.Point(274, 290);
            this.cbActiveVbecsUser.Name = "cbActiveVbecsUser";
            this.cbActiveVbecsUser.Size = new System.Drawing.Size(130, 20);
            this.cbActiveVbecsUser.TabIndex = 0;
            this.cbActiveVbecsUser.Text = "Active VBECS User?";
            this.cbActiveVbecsUser.CheckedChanged += new System.EventHandler(this.DataEntryFields_DataChanged);
            // 
            // btnUserSave
            // 
            this.btnUserSave.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
            this.btnUserSave.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((byte)(0)));
            this.btnUserSave.Image = null;
            this.btnUserSave.Location = new System.Drawing.Point(414, 288);
            this.btnUserSave.Name = "btnUserSave";
            this.btnUserSave.Size = new System.Drawing.Size(75, 23);
            this.btnUserSave.TabIndex = 11;
            this.btnUserSave.Click += new System.EventHandler(this.btnUserSave_Click);
            // 
            // btnUserClear
            // 
            this.btnUserClear.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
            this.btnUserClear.CausesValidation = false;
            this.btnUserClear.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((byte)(0)));
            this.btnUserClear.Image = null;
            this.btnUserClear.Location = new System.Drawing.Point(492, 288);
            this.btnUserClear.Name = "btnUserClear";
            this.btnUserClear.Size = new System.Drawing.Size(75, 23);
            this.btnUserClear.TabIndex = 9;
            this.btnUserClear.Click += new System.EventHandler(this.btnUserClear_Click);
            // 
            // grpUserConfiguration
            // 
            this.grpUserConfiguration.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
            | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.grpUserConfiguration.Controls.Add(this.lvUsers);
            this.grpUserConfiguration.Controls.Add(this.cbShowInactiveUsers);
            this.grpUserConfiguration.Location = new System.Drawing.Point(0, 310);
            this.grpUserConfiguration.Name = "grpUserConfiguration";
            this.grpUserConfiguration.Size = new System.Drawing.Size(568, 160);
            this.grpUserConfiguration.TabIndex = 11;
            this.grpUserConfiguration.TabStop = false;
            this.grpUserConfiguration.Text = "VBECS User Configuration";
            // 
            // lvUsers
            // 
            this.lvUsers.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
            | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.lvUsers.CheckBoxes = true;
            this.lvUsers.Columns.AddRange(new System.Windows.Forms.ColumnHeader[] {
            this.colActive,
            this.colUserID,
            this.colUserName,
            this.colDUZ,
            this.colVistAName,
            this.colInitials,
            this.colEmail});
            this.lvUsers.FullRowSelect = true;
            this.lvUsers.HideSelection = false;
            this.lvUsers.Location = new System.Drawing.Point(6, 18);
            this.lvUsers.MultiSelect = false;
            this.lvUsers.Name = "lvUsers";
            this.lvUsers.Size = new System.Drawing.Size(556, 120);
            this.lvUsers.TabIndex = 3;
            this.lvUsers.UseCompatibleStateImageBehavior = false;
            this.lvUsers.View = System.Windows.Forms.View.Details;
            this.lvUsers.ColumnClick += new System.Windows.Forms.ColumnClickEventHandler(this.lvUsers_ColumnClick);
            this.lvUsers.ItemCheck += new System.Windows.Forms.ItemCheckEventHandler(this.lvUsers_ItemCheck);
            this.lvUsers.SelectedIndexChanged += new System.EventHandler(this.lvUsers_SelectedIndexChanged);
            // 
            // colActive
            // 
            this.colActive.Text = "Active";
            this.colActive.Width = 45;
            // 
            // colUserID
            // 
            this.colUserID.Text = "NT User ID";
            this.colUserID.Width = 90;
            // 
            // colUserName
            // 
            this.colUserName.Text = "NT User Name";
            this.colUserName.Width = 95;
            // 
            // colDUZ
            // 
            this.colDUZ.Text = "DUZ";
            this.colDUZ.Width = 40;
            // 
            // colVistAName
            // 
            this.colVistAName.Text = "VistA User Name";
            this.colVistAName.Width = 95;
            // 
            // colInitials
            // 
            this.colInitials.Text = "Initials";
            this.colInitials.Width = 80;
            // 
            // colEmail
            // 
            this.colEmail.Text = "E-mail Address";
            this.colEmail.Width = 120;
            // 
            // cbShowInactiveUsers
            // 
            this.cbShowInactiveUsers.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Left)));
            this.cbShowInactiveUsers.Location = new System.Drawing.Point(6, 140);
            this.cbShowInactiveUsers.Name = "cbShowInactiveUsers";
            this.cbShowInactiveUsers.Size = new System.Drawing.Size(140, 16);
            this.cbShowInactiveUsers.TabIndex = 4;
            this.cbShowInactiveUsers.Text = "Show Inactive Users";
            this.cbShowInactiveUsers.CheckedChanged += new System.EventHandler(this.cbShowInactiveUsers_CheckedChanged);
            // 
            // editVbecsUserDivisionalAccess
            // 
            this.editVbecsUserDivisionalAccess.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.editVbecsUserDivisionalAccess.Location = new System.Drawing.Point(0, 136);
            this.editVbecsUserDivisionalAccess.Name = "editVbecsUserDivisionalAccess";
            this.editVbecsUserDivisionalAccess.Size = new System.Drawing.Size(568, 144);
            this.editVbecsUserDivisionalAccess.TabIndex = 13;
            this.editVbecsUserDivisionalAccess.ControlDataChanged += new System.EventHandler(this.editVbecsUserDivisionalAccess_ControlDataChanged);
            // 
            // EditVbecsUserParameters
            // 
            this.Controls.Add(this.editVbecsUserDivisionalAccess);
            this.Controls.Add(this.grpUserIdentification);
            this.Controls.Add(this.btnUserSave);
            this.Controls.Add(this.btnUserClear);
            this.Controls.Add(this.cbActiveVbecsUser);
            this.Controls.Add(this.grpUserConfiguration);
            this.Name = "EditVbecsUserParameters";
            this.Size = new System.Drawing.Size(570, 472);
            this.VisibleChanged += new System.EventHandler(this.EditVBecsUsersParameters_VisibleChanged);
            ((System.ComponentModel.ISupportInitialize)(this.erpMainProvider)).EndInit();
            ((System.ComponentModel.ISupportInitialize)(this.erpUserNameProvider)).EndInit();
            this.grpUserIdentification.ResumeLayout(false);
            this.grpVistAUser.ResumeLayout(false);
            this.grpVistAUser.PerformLayout();
            this.grpNTUser.ResumeLayout(false);
            this.grpNTUser.PerformLayout();
            this.grpAdditionalInfo.ResumeLayout(false);
            this.grpAdditionalInfo.PerformLayout();
            this.grpUserConfiguration.ResumeLayout(false);
            this.ResumeLayout(false);

		}
		#endregion

		/// <summary>
		/// Load Event
		/// </summary>
		/// <param name="e"></param>
		protected override void OnLoad(EventArgs e)
		{
			base.OnLoad (e);

#if !DESIGN	

			_formInitialized = true;				
#endif
		}

		private void LoadUsersData()
		{			
			using( new WaitCursor() )
				_vbecsUsers = BOL.VbecsUser.GetAllSortedById();

			this.Refresh();

			using( new WaitCursor() )
				_vistAUsers = BOL.VbecsVistAUser.GetAllVistAUsersHashedByDuz();
		}

		private void RepopulateDynamicListControlsAndResetToNoSelection()
		{
			using( new WaitCursor() )
			{
				this.Refresh();
				DataLoadMode.Lock();
			
				PopulateNtUserLoginCombobox();
				PopulateUsersListView();

				ResetToNoUserSelected();
			}

			this.Refresh();
			DataLoadMode.Release();			
		}

		private void PopulateNtUserLoginCombobox()
		{
			cmbNTUserID.BeginUpdate();			

			cmbNTUserID.Items.Clear();
			cmbNTUserID.DisplayMember = "UserNtLoginId";

			foreach( VbecsUser _user in _vbecsUsers )
				if( _user.IsActive || cbShowInactiveUsers.Checked )
					cmbNTUserID.Items.Add( _user );			

			cmbNTUserID.EndUpdate();
		}

		private void PopulateUsersListView()
		{
			lvUsers.BeginUpdate();
			
			lvUsers.Items.Clear();

			foreach( VbecsUser _user in _vbecsUsers )
			{
				if( !cbShowInactiveUsers.Checked && !_user.IsActive )
					continue;
				
				lvUsers.Items.Add( new VbecsUserListViewItem( _user ) );				
			}

			AddDataFoundVbecsUserListViewItemIfNeeded();
			UpdateVbecsUsersListViewLock();
			UpdateListViewColumnWidths();

			lvUsers.EndUpdate();
		}

		private void UpdateListViewColumnWidths()
		{
			if( IsNoDataFoundInListView )
				return;

			VbecsFunction.SetColumnWidths( lvUsers );
		}

		private bool IsNoDataFoundInListView
		{
			get
			{
				return ( lvUsers.Items.Count == 1 && lvUsers.Items[0] is NoDataFoundListViewItem );
			}
		}

		private void UpdateVbecsUsersListViewLock()
		{
			bool _noDataFound = IsNoDataFoundInListView;
			bool _isDisabled = _noDataFound || ( _selectedUser != null && ( _selectedUser.IsDirty || _selectedUser.IsNew ) );
			
			lvUsers.BackColor = _isDisabled ? SystemColors.Control : SystemColors.Window;					

			if( _noDataFound && lvUsers.Items[0] is ThereAreNoActiveUsersListViewItem && !( _selectedUser != null && _selectedUser.IsNew ) ) 
			{
				// No users shown because they're all inactive
				cbShowInactiveUsers.Enabled = true;
				lvUsers.Enabled = false;
			}
			else
			{
				// Depends. There may be users shown, or there may be no users at all. 
				lvUsers.Enabled = cbShowInactiveUsers.Enabled = !_isDisabled;
			}

			// Multicolumn checked list looks bad when displaying "no data found" message, hence adjusting the style if needed. 
			lvUsers.CheckBoxes = !_noDataFound;
			lvUsers.View = _noDataFound ? View.List : View.Details;
		}

		private void AddDataFoundVbecsUserListViewItemIfNeeded()
		{	
			if( lvUsers.Items.Count > 0 )
				return;

			if( _vbecsUsers.Count == 0 )
				lvUsers.Items.Add( new NoUsersDefinedListViewItem() );
			else
				lvUsers.Items.Add( new ThereAreNoActiveUsersListViewItem() );
		}

		private void ResetToNoUserSelected()
		{			
			_selectedUser = null;

			OnDataChanged( true );
			
			editVbecsUserDivisionalAccess.ResetToNoUserSelected();

			ClearUserParamsControls();
			SetUserConfigControlsState( false );

			cmbNTUserID.Enabled = ( cmbNTUserID.Items.Count > 0 );
		}

		private void ClearUserParamsControls()
		{
			DataLoadMode.Lock();

			lvUsers.SelectedItems.Clear();
			cmbNTUserID.SelectedItem = null;

			txtVistaDuz.Text = String.Empty;
			txtVistaUserName.Text = String.Empty;
			txtUserNtName.Text = String.Empty;
			txtUserEmailAddress.Text = String.Empty;
			txtUserInitials.Text = String.Empty;
			cbActiveVbecsUser.Checked = false;
		
			UpdateVbecsUsersListViewLock();

			DataLoadMode.Release();
		}

		private void SetUserConfigControlsState( bool isEnabled )
		{
			grpAdditionalInfo.Enabled = 
			cbActiveVbecsUser.Enabled = 
			btnUserClear.Enabled = 
			btnUserSave.Enabled = 
			grpVistAUser.Enabled = isEnabled;
		}

		private void lvUsers_ColumnClick(object sender, System.Windows.Forms.ColumnClickEventArgs e)
		{
			ColumnClickSorter.ColumnClickHandler( sender, e, true, ColumnClickSorter.ColumnSortType.String );
		}

		private void lvUsers_ItemCheck(object sender, System.Windows.Forms.ItemCheckEventArgs e)
		{

			//CR2600: indicates whether the user is being activated or inactivated
			bool active = ( e.NewValue == CheckState.Checked );
			//CR2600 end
			if( DataLoadMode.IsLocked )
				return;

			this.ParentEditUserForm.ReleaseAllLocksFromControl();

			using( new WaitCursor() )
			{
				VbecsUserListViewItem _lvi = (VbecsUserListViewItem)lvUsers.Items[e.Index];
				_lvi.VbecsUser.IsActive = ( active );

				if( _selectedUser == _lvi.VbecsUser )
				{					
					DataLoadMode.Lock();
					cbActiveVbecsUser.Checked = _lvi.VbecsUser.IsActive;
					DataLoadMode.Release();
				}
			
				//Ensure the user that has been unchecked is not the only level 6 user
				bool level6UserPresent = true;
				
				if (!_lvi.VbecsUser.IsActive)
				{
					foreach (Division d in _lvi.VbecsUser.DivisionsAndRoles.AllDivisions)
					{
						if (d.IsActive)
						{
							if (!ActiveLevel6UserPresent(_lvi.VbecsUser.VbecsUserGuid,d.DivisionCode))
							{
								level6UserPresent = false;
								break;
							}	
						}
					}
				}
				
				if (level6UserPresent)
				{
					//CR2600
					if( gov.va.med.vbecs.GUI.GuiMessenger.ShowMessageBox( this, Common.StrRes.ConfMsg.MUC03.ChangeUserActivation(e.NewValue == CheckState.Checked ? "activate" : "inactivate", _lvi.VbecsUser.UserName )) == DialogResult.Yes	)
					{
					//CR2600 end
						if (!SaveUser( _lvi.VbecsUser ))
						{
							DataLoadMode.Lock();
							e.NewValue = e.CurrentValue;
							cbActiveVbecsUser.Checked = (e.CurrentValue == CheckState.Checked);
							DataLoadMode.Release();
							return;
						}

                        this.ParentEditUserForm.ReleaseAllLocksFromControl();       //MUC_03 fix - CR3311

						DataLoadMode.Lock();
						_lvi.SyncWithDataObject();
						DataLoadMode.Release();
					}
					//CR2600
					else
					{
						// User decided not to save, so reset ListView back to what it was.
						if (active)
						{
							e.NewValue = CheckState.Unchecked;
							_lvi.VbecsUser.IsActive = false;
						}
						else
						{
							e.NewValue = CheckState.Checked;
							_lvi.VbecsUser.IsActive = true;
						}
					}
					//CR2600 end
				}
				else
				{
					gov.va.med.vbecs.GUI.GuiMessenger.ShowMessageBox( Common.StrRes.SysErrMsg.UC019.RemoveLastAdminUser() );
					DataLoadMode.Lock();
					e.NewValue = e.CurrentValue;
					cbActiveVbecsUser.Checked = (e.CurrentValue == CheckState.Checked);
					DataLoadMode.Release();
				}			

					SelectUser(_lvi.VbecsUser);		//Added to place user back in Edit mode CR2315 (allows editVbecsUserDivisionAccess to have control of objects)	
					
			}
		}

		private void cbShowInactiveUsers_CheckedChanged(object sender, System.EventArgs e)
		{
			RepopulateDynamicListControlsAndResetToNoSelection();
		}

		private void btnVistAUsers_Click(object sender, System.EventArgs e)
		{
			dlgVistAUsers _dlg = new dlgVistAUsers( _vistAUsers.Values );
			
			if( ResultAfterHostRefresh( _dlg.ShowDialog( this ) != DialogResult.OK ) )
				return;
			
			bool _updateDivisionalAccess = ( _dlg.SelectedVistAUser.Duz != _selectedUser.UserDuz );

			txtVistaDuz.Text = _dlg.SelectedVistAUser.Duz;
			txtVistaUserName.Text = _dlg.SelectedVistAUser.UserName;
			txtUserInitials.Text = _dlg.SelectedVistAUser.Initials;

			OnDataChanged( false );
	
			if( _updateDivisionalAccess )
				DisplaySelectedUserDivisionalAccess();
		}

		private bool ConfirmAbandonmentOfChangesIfNeeded()
		{
			if( _selectedUser == null || !( _selectedUser.IsDirty || _selectedUser.IsNew ) )
				return true;

			return ResultAfterHostRefresh( GuiMessenger.ShowMessageBox( StrRes.ConfMsg.MUC03.UserDataNotSaved() ) == DialogResult.Yes );
		}

		private void btnUserClear_Click(object sender, EventArgs e)
		{
			if( !ConfirmAbandonmentOfChangesIfNeeded() )
				return;

			SelectUser( null );

			this.ParentEditUserForm.ReleaseAllLocksFromControl();
		}

		private void cmbNTUserID_SelectedIndexChanged(object sender, EventArgs e)
		{
			if( DataLoadMode.IsLocked )
				return;

			this.Refresh();

			if( ConfirmAbandonmentOfChangesIfNeeded() )
				SelectUser( (VbecsUser)cmbNTUserID.SelectedItem );

			DataLoadMode.Lock();
				cmbNTUserID.SelectedItem = _selectedUser;
			DataLoadMode.Release();
		}

		private void lvUsers_SelectedIndexChanged(object sender, EventArgs e)
		{
			if( DataLoadMode.IsLocked ) 
				return;

			VbecsUserListViewItem _selectedItem = GetSelectedListViewItem();

			SelectUser( _selectedItem != null ? _selectedItem.VbecsUser : null );			
		}

		private void DataEntryFields_DataChanged( object sender, EventArgs e )
		{
			if( DataLoadMode.IsLocked )
				return;

			OnDataChanged( false );
		}

		private void SelectUser( VbecsUser userToSelect )
		{
			VbecsUser _previouslySelectedUser = _selectedUser;

			_selectedUser = userToSelect;

            //Defect 251166 - start with the icon off - this is a catchall
            this.erpUserNameProvider.SetError(this.txtUserNtName, String.Empty);

			using( new WaitCursor() )
			{
			    if (_selectedUser == null)
			        ResetToNoUserSelected();
			    else
			    {
			    	//<Defect 251166> - User Name is read-only, so we can't use broken rules, so we bend over and grab our ankles..
			        bool adNameChanged = (_selectedUser.UserName != _selectedUser.UserNameFromAD);
                    _selectedUser.UserName = _selectedUser.UserNameFromAD;	//another catchall, it will only vary on "Edit"
                    bool userNameRulesAlreadyBroken = (_selectedUser.IsRuleBroken(VbecsBrokenRules.VbecsUser.UserNameTooLong) || _selectedUser.IsRuleBroken(VbecsBrokenRules.VbecsUser.UserNtNameNotSet));
                    //avoid setting 2 errorproviders at once - the broken rules (size > 50 or = 0) should take presidents...
                    if (adNameChanged && !userNameRulesAlreadyBroken) this.erpUserNameProvider.SetError(this.txtUserNtName, Common.StrRes.Tooltips.MUC03.NTUserNameChanged().ResString);
					//</Defect 251166>
					
			        DisplaySelectedNonNullUser();

			    }

			    UpdateAfterChange( _previouslySelectedUser );
				OnDataChanged( true );
			}			

		}

		private void DisplaySelectedNonNullUser()
		{
			DataLoadMode.Lock();			

			cmbNTUserID.SelectedItem = _selectedUser;
			
			DisplayUserDetails();						
			DisplaySelectUserInListView();			
			DisplaySelectedUserDivisionalAccess();
			SetUserConfigControlsState( true );

			DataLoadMode.Release();
		}

		private void DisplayUserDetails()
		{
			txtVistaDuz.Text = _selectedUser.UserDuz;
			txtVistaUserName.Text = _selectedUser.VistAUserName;
			txtUserEmailAddress.Text = _selectedUser.UserEmail;
			txtUserInitials.Text = _selectedUser.UserInitials;
			txtUserNtName.Text = _selectedUser.UserName;
			cbActiveVbecsUser.Checked = _selectedUser.IsActive;			
		}

		private void DisplaySelectedUserDivisionalAccess()
		{
			editVbecsUserDivisionalAccess.SelectUser( _selectedUser.UserDuz != null ? (VbecsVistAUser)_vistAUsers[ _selectedUser.UserDuz ] : null, _selectedUser );
		}

		private void DisplaySelectUserInListView()
		{
			VbecsUserListViewItem _lvi = VbecsUserListViewItem.GetListViewItemByVbecsUser( _selectedUser );
				
			if( GetSelectedListViewItem() != _lvi )
				lvUsers.SelectedItems.Clear();
			
			if( _lvi == null )
				return;
		
			_lvi.Selected = true;
			_lvi.EnsureVisible();
		}

		private VbecsUserListViewItem GetSelectedListViewItem()
		{
			return lvUsers.SelectedItems.Count == 1 ? (VbecsUserListViewItem)lvUsers.SelectedItems[0] : null;
		}

		private void OnDataChanged( bool dontTransferUserInput )
		{
			if( !dontTransferUserInput )
				TransferInputToBusinessObject();			

			_vbecsUserValidator.Validate( _selectedUser );

			SetUpdateButtonState();
			UpdateVbecsUsersListViewLock();
		}

		private void TransferInputToBusinessObject()
		{
			_selectedUser.UserDuz = StUiNullConvert.ToString( txtVistaDuz.Text, true );
			_selectedUser.VistAUserName = StUiNullConvert.ToString( txtVistaUserName.Text, true );
			_selectedUser.UserEmail = StUiNullConvert.ToString( txtUserEmailAddress.Text, true );
			_selectedUser.UserInitials = StUiNullConvert.ToString( txtUserInitials.Text, true );
			_selectedUser.IsActive = cbActiveVbecsUser.Checked;
		}

		private void btnWindowsUsers_Click(object sender, System.EventArgs e)
		{
            DlgWindowsUsers _dlg = new DlgWindowsUsers(_selectedUser);

			if( ResultAfterHostRefresh( _dlg.ShowDialog( this ) != DialogResult.OK ) )
				return;

			VbecsUser _userToSelect = (VbecsUser)_vbecsUsers.GetByObjectID( _dlg.SelectedWindowsUser.LoginId );

			if( ( _selectedUser != null && _selectedUser == _userToSelect ) || !ConfirmAbandonmentOfChangesIfNeeded() )
				return;			

			if( _userToSelect == null )
			{				
				_userToSelect = AddNewVbecsUserBasedOnWindowsUser( _dlg.SelectedWindowsUser );				
				cmbNTUserID.Items.Add( _userToSelect );
			}

			if( !( _userToSelect.IsNew || _userToSelect.IsActive || cbShowInactiveUsers.Checked ) )
				cbShowInactiveUsers.Checked = true;

		    SelectUser( _userToSelect );
		}

		private VbecsUser AddNewVbecsUserBasedOnWindowsUser( VbecsWindowsUser winUser )
		{
			if( winUser == null )
				throw( new ArgumentNullException( "winUser" ) );

			VbecsUser _newUser = new VbecsUser();
			_newUser.UserNTLoginID = winUser.LoginId;
			_newUser.UserName = winUser.FullName;

			return _newUser;
		}

		private void CheckForVistaInactivatedUsers()
		{		
			IList usersInactivatedInVista = GetActiveVbecsUsersInactivatedInVistA();

			if( usersInactivatedInVista.Count == 0 )
				return;

			dlgInactivateUsers dlg = new dlgInactivateUsers( usersInactivatedInVista );

			if( ResultAfterHostRefresh( dlg.ShowDialog( this ) != DialogResult.Yes ) )
				return;
			
			using( new WaitCursor() )
			{			
				InactivateUsersInVbecs( usersInactivatedInVista );

				// Need to reload data to display correctly. 
				// Full reload is needed to update rowversion.
				ReloadUsersInList( usersInactivatedInVista ); 
			}
		}

		private void InactivateUsersInVbecs( IList usersToInactivate )
		{
			if( usersToInactivate == null )
				throw( new ArgumentNullException( "usersToInactivate" ) );

			try
			{
				foreach (VbecsUser user in usersToInactivate)
				{
					foreach (Division d in user.DivisionsAndRoles.AllDivisions)
					{
						if (d.IsActive)
						{
							if (!ActiveLevel6UserPresent(user.VbecsUserGuid,d.DivisionCode))
							{
								//usersToInactivate.Remove(user);
								gov.va.med.vbecs.GUI.GuiMessenger.ShowMessageBox( Common.StrRes.SysErrMsg.UC019.RemoveLastAdminUser() );
								return;
							}
						}
					}
				}
				
				if( LockUsers( usersToInactivate ) )
					VbecsUser.InactivateUsers( usersToInactivate, UpdateFunction.MUC03_EditVbecsUserParameters );
			}
			finally
			{
				UnlockUsers( usersToInactivate );
			}
		}

		private void ReloadUsersInList( IList usersToReload )
		{
			if( usersToReload == null )
				throw( new ArgumentNullException( "usersToReload" ) );

			foreach( VbecsUser _user in usersToReload )
				_user.RefreshFromDb();
		}
		
		private bool LockUsers( IList users )
		{
			if( users == null )
				throw( new ArgumentNullException( "users" ) );
						
			return ParentEditUserForm.SetLocksFromControl( GetUserGuidsFromUserList( users ) );
		}

		private void UnlockUsers( IList users )
		{
			if( users == null )
				throw( new ArgumentNullException( "users" ) );

			ParentEditUserForm.ReleaseLocksFromControl( GetUserGuidsFromUserList( users ) );
		}

		private Guid[] GetUserGuidsFromUserList( IList users )
		{
			if( users == null )
				throw( new ArgumentNullException( "users" ) );

			Guid[] userGuids = new Guid[ users.Count ];

			for( int i = 0; i < users.Count; i++ )
				userGuids[i] = ((VbecsUser)users[i]).VbecsUserGuid;

			return userGuids;
		}

		private DlgEditUserParameters ParentEditUserForm
		{
			get
			{
				return ((DlgEditUserParameters)ParentForm);
			}
		}

		private IList GetActiveVbecsUsersInactivatedInVistA()
		{
			using( new WaitCursor() )
			{
				ArrayList _result = new ArrayList();

				foreach( VbecsUser _vbecsUser in _vbecsUsers )
					if( _vbecsUser.IsActive && !_vistAUsers.ContainsKey( _vbecsUser.UserDuz ) )
						_result.Add( _vbecsUser );

				return _result;
			}
		}

		private void SetUpdateButtonState()
		{
			//CR2315 - don't enable SAVE if no divisions are attached to the user (active or inactive)
			btnUserSave.Enabled = ( _selectedUser != null ) && ( _selectedUser.IsDirty || _selectedUser.IsNew ) && _selectedUser.IsValid && this.editVbecsUserDivisionalAccess.UserDivisionRolesCount > 0;
		}

		private void btnUserSave_Click(object sender, EventArgs e)
		{	
			if( GuiMessenger.ShowMessageBox( this, StrRes.ConfMsg.MUC03.OkToSaveUserChanges() ) != DialogResult.Yes	)
				return;

			this.Refresh();

			using( new WaitCursor() )
			{
				bool level6UserPresent = true;

				DivisionCollection divisions = new DivisionCollection();

				foreach (Division d in _selectedUser.DivisionsAndRoles.AllDivisions)
				{
					if (d.IsActive)
					{
						level6UserPresent = ActiveLevel6UserPresent(_selectedUser.VbecsUserGuid,
							d.DivisionCode);

						if (!level6UserPresent && 
							(_selectedUser.DivisionsAndRoles[d.DivisionCode].Role.UserRoleId != UserRoleType.AdministratorSupervisor ||
							(_selectedUser.DivisionsAndRoles[d.DivisionCode].Role.UserRoleId == UserRoleType.AdministratorSupervisor && !_selectedUser.DivisionsAndRoles[d.DivisionCode].IsActive) ||
							!cbActiveVbecsUser.Checked))
						{
							divisions.Add(d);
						}
					}
				}
				
				if (divisions.Count < 1)
				{
					bool _addingNewUser = _selectedUser.IsNew;
				
					if (!SaveUser( _selectedUser )) return;

					if( !_selectedUser.IsValid )
					{
						OnDataChanged( true );
					}
					else
					{
						UpdateAfterUserSave( _addingNewUser );
						SelectUser( _selectedUser );
					}
				}
				else
				{
					System.Text.StringBuilder divisionList = new System.Text.StringBuilder();

					foreach (Division d in divisions)
					{
						divisionList.Append(d.DivisionName + ", ");
					}

					divisionList.Remove(divisionList.Length -2, 2);

					GuiMessenger.ShowMessageBox( Common.StrRes.SysErrMsg.UC019.RemoveLastAdminUserDivisions(divisionList.ToString()) );
					btnUserSave.Enabled = false;
				}
			}
			
			this.ParentEditUserForm.ReleaseAllLocksFromControl();
		}

		private void UpdateAfterUserSave( bool newUserAdded )
		{
			DataLoadMode.Lock();

			if( newUserAdded )
			{
				_vbecsUsers.Add( _selectedUser );

				EnsureThereAreNoListViewDataNotFoundItems();
				lvUsers.Items.Add( new VbecsUserListViewItem( _selectedUser ) );
				UpdateVbecsUsersListViewLock();
			}
			else
			{
				GetSelectedListViewItem().SyncWithDataObject();
			}

			DataLoadMode.Release();
		}

		private void UpdateAfterChange( VbecsUser previouslySelectedUser )
		{
			if( previouslySelectedUser == null )
				return;
			
			DataLoadMode.Lock();

			if( !previouslySelectedUser.IsNew )
			{
				RestoreUserObjectFromDb( previouslySelectedUser );
				VbecsUserListViewItem.GetListViewItemByVbecsUser( previouslySelectedUser ).SyncWithDataObject();
			}
			else
			{
				cmbNTUserID.Items.Remove( previouslySelectedUser );
			}

			DataLoadMode.Release();
		}

		private void EnsureThereAreNoListViewDataNotFoundItems()
		{
			if( lvUsers.Items.Count == 1 && lvUsers.Items[0] is NoDataFoundListViewItem )
				lvUsers.Items.RemoveAt( 0 );
		}

		private void RestoreUserObjectFromDb( VbecsUser user )
		{
			if( user == null )
				throw( new ArgumentNullException( "user" ) );

			if( !user.IsDirty )
				return;

			using( new WaitCursor() )
				user.RefreshFromDb();
		}
 
		private bool SaveUser( VbecsUser user )
		{
			if( user == null )
				throw( new ArgumentNullException( "user" ) );

			System.Guid[] stasStinx = {user.VbecsUserGuid};

			bool lockingAchieved = ParentEditUserForm.SetLocksFromControl(stasStinx);

			if (lockingAchieved)
			{
                try     //MUC_03 fix - CR3311
                {
                    user.Persist(UpdateFunction.MUC03_EditVbecsUserParameters);
                }
                catch (Common.RowVersionException)
                {
                    this.btnUserSave.RowVersionException = true;		        //CR2077
                    GuiMessenger.ShowMessageBox(Common.StrRes.SysErrMsg.Common.DataWasNotSavedBecauseOfRowversionViolation());
                    return (false);
                }
			}
			return(lockingAchieved);
		}

		/// <summary>
		/// cbActiveVbecsUser_CheckedChanged removed per CR2315 (event now attached to DataEntryFields_DataChanged event)
		/// </summary>
		/// <returns></returns>
		public bool ConfirmExit()
		{
			return ConfirmAbandonmentOfChangesIfNeeded();
		}

		private void editVbecsUserDivisionalAccess_ControlDataChanged(object sender, EventArgs e)
		{
			OnDataChanged( true );
		}

		private void EditVBecsUsersParameters_VisibleChanged(object sender, EventArgs e)
		{
			if (_formInitialized)
			{
				_formInitialized = false;

				LoadUsersData();

				CheckForVistaInactivatedUsers();
				RepopulateDynamicListControlsAndResetToNoSelection();
			}
		}

		/// <summary>
		/// ActiveLevel6UserPresent
		/// </summary>
		/// <param name="userGuid"></param>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public bool ActiveLevel6UserPresent(Guid userGuid, string divisionCode)
		{
			foreach (Object u in _vbecsUsers)
			{
				VbecsUser user = u as VbecsUser;

				if (user != null)
				{
					//Do not include the user being updated in this check.
					if (user.VbecsUserGuid == userGuid)
					{
						continue;
					}

					//Check to see if the user is in the same division. If we get null for the divisionCode 
					//we know the user is not in this division and are not concerned with this user.
					if (user.DivisionsAndRoles[divisionCode] != null)
					{
						//Now make sure the Division is active, roleType is an Admin, and the User is Active.
						if (user.DivisionsAndRoles[divisionCode].IsActive &&
							user.DivisionsAndRoles[divisionCode].Role.UserRoleId 
							== Common.UserRoleType.AdministratorSupervisor &&
							user.IsActive)
						{
							return true;
						}
					}
				}
			}

			return false;
		}

		/// <summary>
		/// ActiveLevel6UserPresent
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <returns></returns>
		public bool ActiveLevel6UserPresent(string divisionCode)
		{
			foreach (Object u in _vbecsUsers)
			{
				VbecsUser user = u as VbecsUser;

				if (user != null)
				{
					//Check to see if the user is in the same division. If we get null for the divisionCode 
					//we know the user is not in this division and are not concerned with this user.
					if (user.DivisionsAndRoles[divisionCode] != null)
					{
						//Now make sure the Division is active, roleType is an Admin, and the User is Active.
						if (user.DivisionsAndRoles[divisionCode].IsActive &&
							user.DivisionsAndRoles[divisionCode].Role.UserRoleId 
							== Common.UserRoleType.AdministratorSupervisor &&
							user.IsActive)
						{
							return true;
						}
					}
				}
			}

			return false;
		}
	}
}